import { isPromise } from './jsutils/isPromise.js';
import type { PromiseOrValue } from './jsutils/PromiseOrValue.js';

import type { GraphQLError } from './error/GraphQLError.js';

import type { DocumentNode } from './language/ast.js';
import type { ParseOptions } from './language/parser.js';
import type { Source } from './language/source.js';

import type { GraphQLSchema } from './type/schema.js';
import { validateSchema } from './type/validate.js';

import type { ValidationOptions } from './validation/validate.js';
import type { ValidationRule } from './validation/ValidationContext.js';

import type { ExecutionArgs } from './execution/execute.js';
import type { ExecutionResult } from './execution/Executor.js';

import type { GraphQLHarness } from './harness.js';
import { defaultHarness } from './harness.js';

/**
 * This is the primary entry point function for fulfilling GraphQL operations
 * by parsing, validating, and executing a GraphQL document along side a
 * GraphQL schema.
 *
 * More sophisticated GraphQL servers, such as those which persist queries,
 * may wish to separate the validation and execution phases to a static time
 * tooling step, and a server runtime step.
 *
 * This function does not support incremental delivery (`@defer` and `@stream`).
 *
 * Accepts either an object with named arguments, or individual arguments:
 *
 * schema:
 *    The GraphQL type system to use when validating and executing a query.
 * source:
 *    A GraphQL language formatted string representing the requested operation.
 * rootValue:
 *    The value provided as the first argument to resolver functions on the top
 *    level type (e.g. the query object type).
 * contextValue:
 *    The context value is provided as an argument to resolver functions after
 *    field arguments. It is used to pass shared information useful at any point
 *    during executing this query, for example the currently logged in user and
 *    connections to databases or other services.
 * variableValues:
 *    A mapping of variable name to runtime value to use for all variables
 *    defined in the requestString.
 * operationName:
 *    The name of the operation to use if requestString contains multiple
 *    possible operations. Can be omitted if requestString contains only
 *    one operation.
 * fieldResolver:
 *    A resolver function to use when one is not provided by the schema.
 *    If not provided, the default field resolver is used (which looks for a
 *    value or method on the source value with the field's name).
 * typeResolver:
 *    A type resolver function to use when none is provided by the schema.
 *    If not provided, the default type resolver is used (which looks for a
 *    `__typename` field or alternatively calls the `isTypeOf` method).
 */
export interface GraphQLArgs
  extends ParseOptions, ValidationOptions, Omit<ExecutionArgs, 'document'> {
  harness?: GraphQLHarness | undefined;
  source: string | Source;
  rules?: ReadonlyArray<ValidationRule> | undefined;
}

export function graphql(args: GraphQLArgs): Promise<ExecutionResult> {
  // Always return a Promise for a consistent API.
  return new Promise((resolve) => resolve(graphqlImpl(args)));
}

/**
 * The graphqlSync function also fulfills GraphQL operations by parsing,
 * validating, and executing a GraphQL document along side a GraphQL schema.
 * However, it guarantees to complete synchronously (or throw an error) assuming
 * that all field resolvers are also synchronous.
 */
export function graphqlSync(args: GraphQLArgs): ExecutionResult {
  const result = graphqlImpl(args);

  // Assert that the execution was synchronous.
  if (isPromise(result)) {
    throw new Error('GraphQL execution failed to complete synchronously.');
  }

  return result;
}

function graphqlImpl(args: GraphQLArgs): PromiseOrValue<ExecutionResult> {
  const harness = args.harness ?? defaultHarness;
  const { schema, source } = args;

  // Validate Schema
  const schemaValidationErrors = validateSchema(schema);
  if (schemaValidationErrors.length > 0) {
    return { errors: schemaValidationErrors };
  }

  // Parse
  let document;
  try {
    document = harness.parse(source, args);
  } catch (syntaxError) {
    return { errors: [syntaxError] };
  }

  if (isPromise(document)) {
    return document.then(
      (resolvedDocument) =>
        validateAndExecute(harness, args, schema, resolvedDocument),
      (syntaxError: unknown) => ({ errors: [syntaxError as GraphQLError] }),
    );
  }

  return validateAndExecute(harness, args, schema, document);
}

function validateAndExecute(
  harness: GraphQLHarness,
  args: GraphQLArgs,
  schema: GraphQLSchema,
  document: DocumentNode,
): PromiseOrValue<ExecutionResult> {
  // Validate
  const validationResult = harness.validate(schema, document, args.rules, args);

  if (isPromise(validationResult)) {
    return validationResult.then((resolvedValidationResult) =>
      checkValidationAndExecute(
        harness,
        args,
        resolvedValidationResult,
        document,
      ),
    );
  }

  return checkValidationAndExecute(harness, args, validationResult, document);
}

function checkValidationAndExecute(
  harness: GraphQLHarness,
  args: GraphQLArgs,
  validationResult: ReadonlyArray<GraphQLError>,
  document: DocumentNode,
): PromiseOrValue<ExecutionResult> {
  if (validationResult.length > 0) {
    return { errors: validationResult };
  }

  // Execute
  return harness.execute({ ...args, document });
}
