/*
   Copyright 2020 Docker Compose CLI authors

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package compose

import (
	"strings"
	"testing"
	"time"

	"github.com/moby/moby/api/types/container"
	"github.com/moby/moby/client"
	"go.uber.org/mock/gomock"
	"gotest.tools/v3/assert"

	compose "github.com/docker/compose/v5/pkg/api"
	"github.com/docker/compose/v5/pkg/utils"
)

func TestStopTimeout(t *testing.T) {
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	api, cli := prepareMocks(mockCtrl)
	tested, err := NewComposeService(cli)
	assert.NilError(t, err)

	api.EXPECT().ContainerList(gomock.Any(), projectFilterListOpt(false)).Return(
		client.ContainerListResult{
			Items: []container.Summary{
				testContainer("service1", "123", false),
				testContainer("service1", "456", false),
				testContainer("service2", "789", false),
			},
		}, nil)
	api.EXPECT().VolumeList(
		gomock.Any(),
		client.VolumeListOptions{
			Filters: projectFilter(strings.ToLower(testProject)),
		}).
		Return(client.VolumeListResult{}, nil)
	api.EXPECT().NetworkList(gomock.Any(), client.NetworkListOptions{Filters: projectFilter(strings.ToLower(testProject))}).
		Return(client.NetworkListResult{}, nil)

	timeout := 2 * time.Second
	stopConfig := client.ContainerStopOptions{Timeout: utils.DurationSecondToInt(&timeout)}
	api.EXPECT().ContainerStop(gomock.Any(), "123", stopConfig).Return(client.ContainerStopResult{}, nil)
	api.EXPECT().ContainerStop(gomock.Any(), "456", stopConfig).Return(client.ContainerStopResult{}, nil)
	api.EXPECT().ContainerStop(gomock.Any(), "789", stopConfig).Return(client.ContainerStopResult{}, nil)

	err = tested.Stop(t.Context(), strings.ToLower(testProject), compose.StopOptions{
		Timeout: &timeout,
	})
	assert.NilError(t, err)
}
