/*
   Copyright 2020 Docker Compose CLI authors

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package compose

import (
	"context"
	"fmt"
	"io"
	"maps"
	"slices"
	"strings"
	"time"

	"github.com/containerd/platforms"
	"github.com/docker/cli/cli/command"
	"github.com/docker/go-units"
	"github.com/moby/moby/client/pkg/stringid"
	"github.com/spf13/cobra"

	"github.com/docker/compose/v5/cmd/formatter"
	"github.com/docker/compose/v5/pkg/api"
	"github.com/docker/compose/v5/pkg/compose"
)

type imageOptions struct {
	*ProjectOptions
	Quiet  bool
	Format string
}

func imagesCommand(p *ProjectOptions, dockerCli command.Cli, backendOptions *BackendOptions) *cobra.Command {
	opts := imageOptions{
		ProjectOptions: p,
	}
	imgCmd := &cobra.Command{
		Use:   "images [OPTIONS] [SERVICE...]",
		Short: "List images used by the created containers",
		RunE: Adapt(func(ctx context.Context, args []string) error {
			return runImages(ctx, dockerCli, backendOptions, opts, args)
		}),
		ValidArgsFunction: completeServiceNames(dockerCli, p),
	}
	imgCmd.Flags().StringVar(&opts.Format, "format", "table", "Format the output. Values: [table | json]")
	imgCmd.Flags().BoolVarP(&opts.Quiet, "quiet", "q", false, "Only display IDs")
	return imgCmd
}

func runImages(ctx context.Context, dockerCli command.Cli, backendOptions *BackendOptions, opts imageOptions, services []string) error {
	projectName, err := opts.toProjectName(ctx, dockerCli)
	if err != nil {
		return err
	}

	backend, err := compose.NewComposeService(dockerCli, backendOptions.Options...)
	if err != nil {
		return err
	}
	images, err := backend.Images(ctx, projectName, api.ImagesOptions{
		Services: services,
	})
	if err != nil {
		return err
	}

	if opts.Quiet {
		ids := []string{}
		for _, img := range images {
			id := img.ID
			if i := strings.IndexRune(img.ID, ':'); i >= 0 {
				id = id[i+1:]
			}
			if !slices.Contains(ids, id) {
				ids = append(ids, id)
			}
		}
		for _, img := range ids {
			_, _ = fmt.Fprintln(dockerCli.Out(), img)
		}
		return nil
	}
	if opts.Format == "json" {

		type img struct {
			ID            string     `json:"ID"`
			ContainerName string     `json:"ContainerName"`
			Repository    string     `json:"Repository"`
			Tag           string     `json:"Tag"`
			Platform      string     `json:"Platform"`
			Size          int64      `json:"Size"`
			Created       *time.Time `json:"Created,omitempty"`
			LastTagTime   time.Time  `json:"LastTagTime,omitzero"`
		}
		// Convert map to slice
		var imageList []img
		for ctr, i := range images {
			lastTagTime := i.LastTagTime
			imageList = append(imageList, img{
				ContainerName: ctr,
				ID:            i.ID,
				Repository:    i.Repository,
				Tag:           i.Tag,
				Platform:      platforms.Format(i.Platform),
				Size:          i.Size,
				Created:       i.Created,
				LastTagTime:   lastTagTime,
			})
		}
		json, err := formatter.ToJSON(imageList, "", "")
		if err != nil {
			return err
		}
		_, err = fmt.Fprintln(dockerCli.Out(), json)
		return err
	}

	return formatter.Print(images, opts.Format, dockerCli.Out(),
		func(w io.Writer) {
			for _, container := range slices.Sorted(maps.Keys(images)) {
				img := images[container]
				id := stringid.TruncateID(img.ID)
				size := units.HumanSizeWithPrecision(float64(img.Size), 3)
				repo := img.Repository
				if repo == "" {
					repo = "<none>"
				}
				tag := img.Tag
				if tag == "" {
					tag = "<none>"
				}
				created := "N/A"
				if img.Created != nil {
					created = units.HumanDuration(time.Now().UTC().Sub(*img.Created)) + " ago"
				}
				_, _ = fmt.Fprintf(w, "%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
					container, repo, tag, platforms.Format(img.Platform), id, size, created)
			}
		},
		"CONTAINER", "REPOSITORY", "TAG", "PLATFORM", "IMAGE ID", "SIZE", "CREATED")
}
